/* ***************************************************** **
   ch06_iterative_calculations_with_multidimensional_data.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 6.
   Obliczenia iteracyjne z użyciem danych wielowymiarowych
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

-- W przeciwieństwie do innych rozdziałów, w tym kolmny są
-- formatowane ręcznie z użyciem sqlformat ansiconsole

set pagesize 80
set linesize 80
set sqlformat
column generation format 99
column x          format 99
column y          format 99
column alive      format 9999
column cells      format a10
column sum_alives format a10
column nb_alives  format a10

/* -----------------------------------------------------
   Przykładowe fragmenty kodu do rozdziału 6.
   ----------------------------------------------------- */

-- Listing 6.1. Tworzenie generacji zerowej w postaci siatki o wielkości 10×10

truncate table conway_gen_zero;

insert into conway_gen_zero (x, y, alive)
select * from (
   with numbers as (
      select level as n from dual
      connect by level <= 10
   ), grid as (
      select
         x.n as x
       , y.n as y
      from numbers x
      cross join numbers y
   ), start_cells as (
      select  4 x,  4 y from dual union all
      select  5 x,  4 y from dual union all
      select  4 x,  5 y from dual union all
      select  6 x,  6 y from dual union all
      select  7 x,  6 y from dual union all
      select  4 x,  7 y from dual union all
      select  5 x,  7 y from dual union all
      select  6 x,  7 y from dual
   )
   select
      g.x
    , g.y
    , nvl2(sc.x, 1, 0) as alive
   from grid g
   left outer join start_cells sc
      on  sc.x = g.x
      and sc.y = g.y
);

commit;

-- Listing 6.2. Wizualizacja generacji zerowej gry w życie

select
   listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) as cells
from conway_gen_zero
group by y
order by y;

-- Listing 6.3. Wykorzystanie klauzuli model do obliczania liczby żywych sąsiadów poszczególnych komórek 

select *
from conway_gen_zero
model
dimension by (
   x, y
)
measures (
   alive
 , 0 as sum_alive
 , 0 as nb_alive
)
ignore nav
rules
(
   sum_alive[any, any] =
      sum(alive)[
         x between cv() - 1 and cv() + 1
       , y between cv() - 1 and cv() + 1
      ]
 , nb_alive[any, any] =
      sum_alive[cv(), cv()] - alive[cv(), cv()]
)
order by x, y;

-- Listing 6.4. Obliczanie liczby żywych sąsiadów za pomocą podzapytania skalarnego

select
   x
 , y
 , alive
 , sum_alive
 , sum_alive - alive as nb_alive
from (
   select
      x
    , y
    , alive
    , (
         select sum(gz2.alive)
         from conway_gen_zero gz2
         where gz2.x between gz.x - 1 and gz.x + 1
         and   gz2.y between gz.y - 1 and gz.y + 1
      ) as sum_alive
   from conway_gen_zero gz
)
order by x, y;

-- Listing 6.5. Inny sposób wyświetlenia informacji o żywych sąsiadach na siatce gry w życie

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      x, y
   )
   measures (
      alive
    , 0 as sum_alive
    , 0 as nb_alive
   )
   ignore nav
   rules
   (
      sum_alive[any, any] =
         sum(alive)[
            x between cv() - 1 and cv() + 1
          , y between cv() - 1 and cv() + 1
         ]
    , nb_alive[any, any] =
         sum_alive[cv(), cv()] - alive[cv(), cv()]
   )
)
select
   listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
 , listagg(sum_alive) within group (order by x) sum_alives
 , listagg(nb_alive ) within group (order by x) nb_alives
from conway
group by y
order by y;

-- Wariant listingu 6.5 na podstawie listingu 6.4 zamiast 6.3

with conway as (
   select
      x
    , y
    , alive
    , sum_alive
    , sum_alive - alive as nb_alive
   from (
      select
         x
       , y
       , alive
       , (
            select sum(gz2.alive)
            from conway_gen_zero gz2
            where gz2.x between gz.x - 1 and gz.x + 1
            and   gz2.y between gz.y - 1 and gz.y + 1
         ) as sum_alive
      from conway_gen_zero gz
   )
)
select
   listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
 , listagg(sum_alive) within group (order by x) sum_alives
 , listagg(nb_alive ) within group (order by x) nb_alives
from conway
group by y
order by y;

-- Listing 6.6. Iteracja przez dwie generacje gry w życie

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      0 as generation
    , x, y
   )
   measures (
      alive
    , 0 as sum_alive
    , 0 as nb_alive
   )
   ignore nav
   rules upsert all iterate (2)
   (
      sum_alive[iteration_number, any, any] =
         sum(alive)[
            generation = iteration_number
          , x between cv() - 1 and cv() + 1
          , y between cv() - 1 and cv() + 1
         ]
    , nb_alive[iteration_number, any, any] =
         sum_alive[iteration_number, cv(), cv()]
           - alive[iteration_number, cv(), cv()]
    , alive[iteration_number + 1, any, any] =
         case nb_alive[iteration_number, cv(), cv()]
            when 2 then alive[iteration_number, cv(), cv()]
            when 3 then 1
            else 0
         end
   )
)
select
   generation
 , listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
 , listagg(sum_alive) within group (order by x) sum_alives
 , listagg(nb_alive ) within group (order by x) nb_alives
from conway
group by generation, y
order by generation, y;

-- Listing 6.7. Skrócona wersja zapytania

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      0 as generation
    , x, y
   )
   measures (
      alive
   )
   ignore nav
   rules upsert all iterate (2)
   (
      alive[iteration_number + 1, any, any] =
         case sum(alive)[
                  generation = iteration_number,
                  x between cv() - 1 and cv() + 1,
                  y between cv() - 1 and cv() + 1
              ] - alive[iteration_number, cv(), cv()]
            when 2 then alive[iteration_number, cv(), cv()]
            when 3 then 1
            else 0
         end
   )
)
select
   generation
 , listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
from conway
group by generation, y
order by generation, y;

-- 25 generacji

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      0 as generation
    , x, y
   )
   measures (
      alive
   )
   ignore nav
   rules upsert all iterate (25)
   (
      alive[iteration_number + 1, any, any] =
         case sum(alive)[
                  generation = iteration_number,
                  x between cv() - 1 and cv() + 1,
                  y between cv() - 1 and cv() + 1
              ] - alive[iteration_number, cv(), cv()]
            when 2 then alive[iteration_number, cv(), cv()]
            when 3 then 1
            else 0
         end
   )
)
select
   generation
 , listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
from conway
group by generation, y
order by generation, y;

-- 50 generacji

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      0 as generation
    , x, y
   )
   measures (
      alive
   )
   ignore nav
   rules upsert all iterate (50)
   (
      alive[iteration_number + 1, any, any] =
         case sum(alive)[
                  generation = iteration_number,
                  x between cv() - 1 and cv() + 1,
                  y between cv() - 1 and cv() + 1
              ] - alive[iteration_number, cv(), cv()]
            when 2 then alive[iteration_number, cv(), cv()]
            when 3 then 1
            else 0
         end
   )
)
select
   generation
 , listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
from conway
group by generation, y
order by generation, y;

-- Listing 6.8. Gra w życie na siatce o wielkości 6×6

truncate table conway_gen_zero;

insert into conway_gen_zero (x, y, alive)
select * from (
   with numbers as (
      select level as n from dual
      connect by level <= 6
   ), grid as (
      select
         x.n as x
       , y.n as y
      from numbers x
      cross join numbers y
   ), start_cells as (
      select  4 x,  2 y from dual union all
      select  2 x,  3 y from dual union all
      select  5 x,  3 y from dual union all
      select  2 x,  4 y from dual union all
      select  5 x,  4 y from dual union all
      select  3 x,  5 y from dual
   )
   select
      g.x
    , g.y
    , nvl2(sc.x, 1, 0) as alive
   from grid g
   left outer join start_cells sc
      on  sc.x = g.x
      and sc.y = g.y
);

commit;

-- 2 generacje pokazują generację zero i dwie identyczne

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      0 as generation
    , x, y
   )
   measures (
      alive
   )
   ignore nav
   rules upsert all iterate (2)
   (
      alive[iteration_number + 1, any, any] =
         case sum(alive)[
                  generation = iteration_number,
                  x between cv() - 1 and cv() + 1,
                  y between cv() - 1 and cv() + 1
              ] - alive[iteration_number, cv(), cv()]
            when 2 then alive[iteration_number, cv(), cv()]
            when 3 then 1
            else 0
         end
   )
)
select
   generation
 , listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
from conway
group by generation, y
order by generation, y;

-- Oscylacja między dwoma stanami, niezależnie od liczby generacji

with conway as (
   select *
   from conway_gen_zero
   model
   dimension by (
      0 as generation
    , x, y
   )
   measures (
      alive
   )
   ignore nav
   rules upsert all iterate (20)
   (
      alive[iteration_number + 1, any, any] =
         case sum(alive)[
                  generation = iteration_number,
                  x between cv() - 1 and cv() + 1,
                  y between cv() - 1 and cv() + 1
              ] - alive[iteration_number, cv(), cv()]
            when 2 then alive[iteration_number, cv(), cv()]
            when 3 then 1
            else 0
         end
   )
)
select
   generation
 , listagg(
      case alive
         when 1 then 'X'
         when 0 then ' '
      end
   ) within group (
      order by x
   ) cells
from conway
group by generation, y
order by generation, y;

/* ***************************************************** */
